'use strict';

/* --------------------------------------------------------------
 customers_table_controller.js 2018-05-16
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2017 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

/**
 * ## Customers Table Controller
 *
 * This controller contains the mapping logic of the customers table.
 *
 * @module Compatibility/customers_table_controller
 */
gx.compatibility.module('customers_table_controller', [gx.source + '/libs/button_dropdown'],

/**  @lends module:Compatibility/customers_table_controller */

function (data) {

    'use strict';

    // ------------------------------------------------------------------------
    // VARIABLES DEFINITION
    // ------------------------------------------------------------------------

    var
    /**
     * Module Selector
     *
     * @var {object}
     */
    $this = $(this),


    /**
     * Default Options
     *
     * @type {object}
     */
    defaults = {
        'adminAccessGranted': false,
        'customerGroupsGranted': false,
        'mailbeezGranted': false,
        'mediafinanzGranted': false,
        'ordersGranted': false
    },


    /**
     * Final Options
     *
     * @var {object}
     */
    options = $.extend(true, {}, defaults, data),


    /**
     * Module Object
     *
     * @type {object}
     */
    module = {},


    /**
     * Reference to the actual file
     *
     * @var {string}
     */
    srcPath = window.location.origin + window.location.pathname,


    /**
     * Query parameter string
     *
     * @type {string}
     */
    queryString = '?' + window.location.search.replace(/\?/, '').replace(/cID=[\d]+/g, '').replace(/action=[\w]+/g, '').concat('&').replace(/&[&]+/g, '&').replace(/^&/g, '');

    // ------------------------------------------------------------------------
    // PRIVATE METHODS
    // ------------------------------------------------------------------------

    /**
     * Map actions for every row in the table.
     *
     * This method will map the actions for each
     * row of the table.
     *
     * @private
     */
    var _mapRowActions = function _mapRowActions() {
        // Iterate over table rows, except the header row
        $('.gx-customer-overview tr').not('.dataTableHeadingRow').each(function () {

            /**
             * Save that "this" scope here
             * @var {object | jQuery}
             */
            var $that = $(this);

            /**
             * Data attributes of current row
             * @var {object}
             */
            var data = $that.data();

            /**
             * Reference to the row action dropdown
             * @var {object | jQuery}
             */
            var $dropdown = $that.find('.js-button-dropdown');

            if ($dropdown.length) {

                // Add click event to the table row and open the
                // customer detail view
                $that.find('.btn-edit').closest('tr').css({
                    cursor: 'pointer'
                }).on('click', function (event) {
                    // Compose the URL and open it
                    var url = [srcPath, '?cID=' + data.rowId, '&action=edit'].join('');
                    if ($(event.target).prop('tagName') === 'TD') {
                        window.open(url, '_self');
                    }
                });

                // Icon behavior - Edit
                $that.find('.btn-edit').css({
                    cursor: 'pointer'
                }).prop('title', jse.core.lang.translate('edit', 'buttons')).on('click', function () {
                    // Compose the URL and open it
                    var url = [srcPath, '?cID=' + data.rowId, '&action=edit'].join('');
                    window.open(url, '_self');
                });

                // Icon behavior - Delete
                if (data.rowId !== 1) {
                    $that.find('.btn-delete').css({
                        cursor: 'pointer'
                    }).prop('title', jse.core.lang.translate('delete', 'buttons')).on('click', function () {
                        // Compose the URL and open it
                        var url = [srcPath, queryString, 'cID=' + data.rowId, '&action=confirm'].join('');
                        window.open(url, '_self');
                    });
                }

                if (data.rowId === 1) {
                    $that.find('.btn-delete').css({
                        opacity: '0.2'
                    });
                }

                // Icon behavior - Orders
                if (options.ordersGranted) {
                    $that.find('.btn-order').css({
                        cursor: 'pointer'
                    }).prop('title', jse.core.lang.translate('BUTTON_ORDERS', 'admin_buttons')).on('click', function () {
                        // Compose the URL and open it
                        var url = [srcPath.replace('customers.php', 'admin.php'), '?' + $.param({
                            do: 'OrdersOverview',
                            filter: {
                                customer: '#' + data.rowId
                            }
                        })].join('');
                        window.open(url, '_self');
                    });
                }

                _mapButtonDropdown($that, $dropdown, data);
            }
        });
    };

    var _mapButtonDropdown = function _mapButtonDropdown($that, $dropdown, data) {
        var actions = ['edit'];

        // Bind dropdown option - Delete
        if (data.rowId !== 1) {
            actions.push('delete');
        }

        actions = actions.concat(['BUTTON_EMAIL', 'BUTTON_IPLOG', 'export_personal_data']);

        if ($that.find('[data-cust-group]').data('custGroup') !== 0) {
            actions.push('delete_personal_data');
        }

        if ($that.find('[data-cust-group]').data('custGroup') !== 0) {
            actions.push('BUTTON_LOGIN_AS_CUSTOMER');
        }

        if (options.customerGroupsGranted) {
            actions.push('BUTTON_STATUS');
        }

        if (options.ordersGranted) {
            actions.push('BUTTON_ORDERS');
            actions.push('BUTTON_NEW_ORDER');
        }

        // Admin rights button
        if ($that.find('[data-cust-group]').data('custGroup') === 0 && options.adminAccessGranted) {
            actions.push('BUTTON_ACCOUNTING');
        }

        // Bind MailBeez dropdown options.
        var mailBeezConversationsSelector = '.contentTable .infoBoxContent a.context_view_button.btn_right';
        if ($(mailBeezConversationsSelector).length && options.mailbeezGranted) {
            actions.push('MAILBEEZ_OVERVIEW');
            actions.push('MAILBEEZ_NOTIFICATIONS');
            actions.push('MAILBEEZ_CONVERSATIONS');
        }

        // Bind Mediafinanz dropdown options.
        var $mediafinanzAction = $('.mediafinanz-creditworthiness');
        if ($mediafinanzAction.length && options.mediafinanzGranted) {
            actions.push('BUTTON_MEDIAFINANZ_CREDITWORTHINESS');
        }

        for (var index in actions) {
            _mapCustomerAction($dropdown, actions[index], data);
        }
    };

    var _mapCustomerAction = function _mapCustomerAction($dropdown, action, data) {
        jse.libs.button_dropdown.mapAction($dropdown, action, _sectionMapping[action], function (event) {
            _executeActionCallback(action, data);
        });
    };

    var _sectionMapping = {
        edit: 'buttons',
        delete: 'buttons',
        BUTTON_STATUS: 'admin_buttons',
        BUTTON_ORDERS: 'admin_buttons',
        BUTTON_EMAIL: 'admin_buttons',
        BUTTON_IPLOG: 'admin_buttons',
        MAILBEEZ_OVERVIEW: 'admin_customers',
        MAILBEEZ_NOTIFICATIONS: 'admin_customers',
        MAILBEEZ_CONVERSATIONS: 'admin_customers',
        BUTTON_MEDIAFINANZ_CREDITWORTHINESS: 'admin_buttons',
        BUTTON_NEW_ORDER: 'admin_buttons',
        BUTTON_ACCOUNTING: 'admin_buttons',
        BUTTON_LOGIN_AS_CUSTOMER: 'admin_customers',
        delete_personal_data: 'admin_customers',
        export_personal_data: 'admin_customers'
    };

    /**
     * Get the corresponding callback
     *
     * @param action
     * @private
     */
    var _executeActionCallback = function _executeActionCallback(action, data) {
        switch (action) {
            case 'edit':
                _editCallback(data);
                break;
            case 'delete':
                _deleteCallback(data);
                break;
            case 'BUTTON_STATUS':
                _customerGroupCallBack(data);
                break;
            case 'BUTTON_ORDERS':
                _ordersCallback(data);
                break;
            case 'BUTTON_EMAIL':
                _emailCallback(data);
                break;
            case 'BUTTON_IPLOG':
                _ipLogCallback(data);
                break;
            case 'MAILBEEZ_OVERVIEW':
                _mailBeezOverviewCallback(data);
                break;
            case 'MAILBEEZ_NOTIFICATIONS':
                _mailBeezNotificationsCallback(data);
                break;
            case 'MAILBEEZ_CONVERSATIONS':
                _mailBeezConversationsCallback(data);
                break;
            case 'BUTTON_MEDIAFINANZ_CREDITWORTHINESS':
                _mediafinanzCreditworthinessCallback(data);
                break;
            case 'BUTTON_NEW_ORDER':
                _newOrderCallback(data);
                break;
            case 'BUTTON_ACCOUNTING':
                _adminRightsCallback(data);
                break;
            case 'BUTTON_LOGIN_AS_CUSTOMER':
                _loginAsCustomerCallback(data);
                break;
            case 'delete_personal_data':
                _deletePersonalDataCallback(data);
                break;
            case 'export_personal_data':
                _exportPersonalDataCallback(data);
                break;
            default:
                throw new Error('Callback not found.');
                break;
        }
    };

    var _editCallback = function _editCallback(data) {
        // Compose the URL and open it
        var url = [srcPath, '?cID=' + data.rowId, '&action=edit'].join('');
        window.open(url, '_self');
    };

    var _deleteCallback = function _deleteCallback(data) {
        // Compose the URL and open it
        var url = [srcPath, queryString, 'cID=' + data.rowId, '&action=confirm'].join('');
        window.open(url, '_self');
    };

    var _customerGroupCallBack = function _customerGroupCallBack(data) {
        // Compose the URL and open it
        var url = [srcPath, queryString, 'cID=' + data.rowId, '&action=editstatus'].join('');
        window.open(url, '_self');
    };

    var _ordersCallback = function _ordersCallback(data) {
        // Compose the URL and open it
        var url = [srcPath.replace('customers.php', 'admin.php'), '?' + $.param({
            do: 'OrdersOverview',
            filter: {
                customer: '#' + data.rowId
            }
        })].join('');
        window.open(url, '_self');
    };

    var _emailCallback = function _emailCallback(data) {
        // Compose the URL and open it
        var url = [srcPath.replace('customers.php', 'mail.php'), '?selected_box=tools', '&customer=' + data.custEmail].join('');
        window.open(url, '_self');
    };

    var _ipLogCallback = function _ipLogCallback(data) {
        // Compose the URL and open it
        var url = [srcPath, queryString, 'cID=' + data.rowId, '&action=iplog'].join('');
        window.open(url, '_self');
    };

    var _mailBeezOverviewCallback = function _mailBeezOverviewCallback(data) {
        var $target = $('.contentTable .infoBoxContent a.context_view_button.btn_left');
        var url = $('.contentTable .infoBoxContent a.context_view_button.btn_left').attr('onclick');
        url = url.replace(/cID=(.*)&/, 'cID=' + data.rowId + '&');
        $('.contentTable .infoBoxContent a.context_view_button.btn_left').attr('onclick', url);
        $target.get(0).click();
    };

    var _mailBeezNotificationsCallback = function _mailBeezNotificationsCallback(data) {
        var $target = $('.contentTable .infoBoxContent a.context_view_button.btn_middle');
        var url = $('.contentTable .infoBoxContent a.context_view_button.btn_middle').attr('onclick');
        url = url.replace(/cID=(.*)&/, 'cID=' + data.rowId + '&');
        $('.contentTable .infoBoxContent a.context_view_button.btn_middle').attr('onclick', url);
        $target.get(0).click();
    };

    var _mailBeezConversationsCallback = function _mailBeezConversationsCallback(data) {
        var $target = $('.contentTable .infoBoxContent a.context_view_button.btn_right');
        var url = $('.contentTable .infoBoxContent a.context_view_button.btn_right').attr('onclick');
        url = url.replace(/cID=(.*)&/, 'cID=' + data.rowId + '&');
        $('.contentTable .infoBoxContent a.context_view_button.btn_right').attr('onclick', url);
        $target.get(0).click();
    };

    var _mediafinanzCreditworthinessCallback = function _mediafinanzCreditworthinessCallback(data) {
        var $target = $('.mediafinanz-creditworthiness');
        var onclickAttribute = $target.attr('onclick');
        // Replace the customer number in the onclick attribute. 
        onclickAttribute = onclickAttribute.replace(/cID=(.*', 'popup')/, 'cID=' + data.rowId + '\', \'popup\'');
        $target.attr('onclick', onclickAttribute);
        $target.trigger('click'); // Trigger the click event in the <a> element. 
    };

    var _newOrderCallback = function _newOrderCallback(data) {
        // Compose the URL and open it
        var url = [srcPath, '?cID=' + data.rowId, '&action=new_order'].join('');
        window.open(url, '_self');
    };

    var _adminRightsCallback = function _adminRightsCallback(data) {
        // Compose the URL and open it
        var url = 'admin.php?do=AdminAccess/editAdmin&id=' + data.rowId;
        window.open(url, '_self');
    };

    var _loginAsCustomerCallback = function _loginAsCustomerCallback(data) {
        // Compose the URL and open it
        var url = [srcPath.replace('customers.php', 'admin.php'), '?do=CustomerLogin&customerId=' + data.rowId + '&pageToken=' + jse.core.config.get('pageToken')].join('');
        window.open(url, '_self');
    };

    var _deletePersonalDataCallback = function _deletePersonalDataCallback(data) {
        // Compose the URL and open it
        var url = [srcPath, queryString, 'cID=' + data.rowId, '&action=delete_personal_data'].join('');
        window.open(url, '_self');
    };

    var _exportPersonalDataCallback = function _exportPersonalDataCallback(data) {
        // Compose the URL and open it
        var url = [srcPath, queryString, 'cID=' + data.rowId, '&action=export_personal_data'].join('');
        window.open(url, '_self');
    };

    // ------------------------------------------------------------------------
    // INITIALIZATION
    // ------------------------------------------------------------------------

    module.init = function (done) {
        // Wait until the buttons are converted to dropdown for every row.
        var interval = setInterval(function () {
            if ($('.js-button-dropdown').length) {
                clearInterval(interval);
                _mapRowActions();
            }
        }, 500);

        done();
    };

    return module;
});
//# sourceMappingURL=data:application/json;charset=utf8;base64,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
